export function add<T extends DateTime | Date | Time>(a: TimeDelta, b: TimeDelta): TimeDelta;
export function add<T extends DateTime | Date | Time>(a: T, b: TimeDelta): T;
export function add<T extends DateTime | Date | Time>(a: TimeDelta, b: T): T;
export function sub<T extends DateTime | Date | Time>(a: TimeDelta, b: TimeDelta): TimeDelta;
export function sub<T extends DateTime | Date | Time>(a: T, b: T): TimeDelta;
export function sub<T extends DateTime | Date | Time>(a: T, b: TimeDelta): T;
/**
 * Negate datetime objects.
 * @param {TimeDelta} a The value.
 * @returns {TimeDelta}
 */
export function neg(a: TimeDelta): TimeDelta;
/**
 * Compare two datetime objects.
 * Return 0 if two are the same, 1 if left side is greater than right,
 * -1 if right side is greater than left.
 * @template {(!TimeDelta|!DateTime|!Date|!Time)} T
 * @param {T} a Left side value.
 * @param {T} b Right side value.
 * @returns {(-1|0|1)}
 */
export function cmp<T extends TimeDelta | DateTime | Date | Time>(a: T, b: T): (-1 | 0 | 1);
/**
 * Convenient function to create {@link TimeDelta} object.
 * @param {ConstructorParameters<typeof TimeDelta>} args
 * @returns {TimeDelta}
 */
export function timedelta(args_0?: {
    days?: number | undefined;
    seconds?: number | undefined;
    microseconds?: number | undefined;
    milliseconds?: number | undefined;
    minutes?: number | undefined;
    hours?: number | undefined;
    weeks?: number | undefined;
} | undefined): TimeDelta;
/**
 * Convenient function to create {@link Date} object.
 * @param {ConstructorParameters<typeof Date>} args
 * @returns {Date}
 */
export function date(year: number, month: number, day: number): Date;
/**
 * Convenient function to create {@link TimeZone} object.
 * @param {ConstructorParameters<typeof TimeZone>} args
 * @returns {TimeZone}
 */
export function timezone(offset: TimeDelta, name?: string | null | undefined): TimeZone;
/**
 * Convenient function to create {@link Time} object.
 * @param {ConstructorParameters<typeof Time>} args
 * @returns {Time}
 */
export function time(hour?: number | undefined, minute?: number | undefined, second?: number | undefined, microsecond?: number | undefined, tzInfo?: TZInfo | null | undefined, fold?: number | undefined): Time;
/**
 * Convenient function to create {@link DateTime} object.
 * @param {ConstructorParameters<typeof DateTime>} args
 * @returns {DateTime}
 */
export function datetime(year: number, month: number, day: number, hour?: number | undefined, minute?: number | undefined, second?: number | undefined, microsecond?: number | undefined, tzInfo?: TZInfo | null | undefined, fold?: number | undefined): DateTime;
/**
 * The smallest year number allowed in a date or datetime object.
 */
export const MINYEAR: 1;
/**
 * The largest year number allowed in a date or datetime object.
 */
export const MAXYEAR: 9999;
/**
 * Represents a duration, the difference between two dates or times.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.timedelta.
 */
export class TimeDelta {
    /**
     * The most negative timedelta object, new TimeDelta(\{days: -999999999\}).
     * @type {!TimeDelta}
     */
    static get min(): TimeDelta;
    /**
     * The most positive timedelta object, new TimeDelta(\{days: 999999999,
     * hours: 23, minutes: 59, seconds: 59, microseconds: 999999\}).
     * @type {!TimeDelta}
     */
    static get max(): TimeDelta;
    /**
     * The smallest possible difference between non-equal timedelta objects,
     * new TimeDelta(\{microseconds: 1\}).
     * @type {!TimeDelta}
    */
    static get resolution(): TimeDelta;
    /**
     * @param {Object} duration An object consisting of duration values.
     * @param {number} [duration.days]
     * @param {number} [duration.seconds]
     * @param {number} [duration.microseconds]
     * @param {number} [duration.milliseconds]
     * @param {number} [duration.minutes]
     * @param {number} [duration.hours]
     * @param {number} [duration.weeks]
     */
    constructor({ days, seconds, microseconds, milliseconds, minutes, hours, weeks, }?: {
        days?: number | undefined;
        seconds?: number | undefined;
        microseconds?: number | undefined;
        milliseconds?: number | undefined;
        minutes?: number | undefined;
        hours?: number | undefined;
        weeks?: number | undefined;
    });
    /**
     * @private
     * @readonly
     */
    private readonly _days;
    /**
     * @private
     * @readonly
     */
    private readonly _seconds;
    /**
     * @private
     * @readonly
     */
    private readonly _microseconds;
    /**
     * Between -999999999 and 999999999 inclusive.
     * @type {number}
     */
    get days(): number;
    /**
     * Between 0 and 86399 inclusive
     * @type {number}
     */
    get seconds(): number;
    /**
     * Between 0 and 999999 inclusive
     * @type {number}
     */
    get microseconds(): number;
    /**
     * Return the total number of seconds contained in the duration.
     * @returns {number}
     */
    totalSeconds(): number;
    /**
     * Return the human-readable string respresentation.
     * @returns {string}
     */
    toString(): string;
    /**
     * Same as totalSeconds().
     * @returns {number}
     */
    valueOf(): number;
}
/**
 * Represents a date (year, month and day) in an idealized calendar.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.date.
 */
export class Date {
    /**
     * Return the Date corresponding to the given standard library Date object.
     * @param {!stdDate} d The standard library Date object.
     * @param {boolean} utc If true, use getUTC***() instead of get***()
     *                      to construct Date.
     * @returns {!Date}
     */
    static fromStdDate(d: stdDate, utc?: boolean): Date;
    /**
     * Return the current local date.
     * @returns {!Date}
     */
    static today(): Date;
    /**
     * Return the Date corresponding to the proleptic Gregorian ordinal, where
     * January 1 of year 1 has ordinal 1.
     * @param {number} ordinal The proleptic Gregorian ordinal.
     * @returns {!Date}
     */
    static fromOrdinal(ordinal: number): Date;
    /**
     * Return a Date corresponding to a dateString given in the format
     * `YYYY-MM-DD` or `YYYYMMDD`.
     * @param {string} dateString The date string.
     * @returns {!Date}
     */
    static fromISOFormat(dateString: string): Date;
    /**
     * The earliest representable date, new Date(MINYEAR, 1, 1).
     * @type {!Date}
     */
    static get min(): Date;
    /**
     * The latest representable date, new Date(MAXYEAR, 12, 31).
     * @type {!Date}
     */
    static get max(): Date;
    /**
     * The smallest possible difference between non-equal date objects,
     * new TimeDelta(\{days: 1\}).
     * @type {!TimeDelta}
     */
    static get resolution(): TimeDelta;
    /**
     * @param {number} year Between MINYEAR and MAXYEAR.
     * @param {number} month Between 1 and 12.
     * @param {number} day Between 1 and the number of days in the given month
     *                     and year.
     */
    constructor(year: number, month: number, day: number);
    /**
     * @private
     * @readonly
     */
    private readonly _year;
    /**
     * @private
     * @readonly
     */
    private readonly _month;
    /**
     * @private
     * @readonly
     */
    private readonly _day;
    /**
     * Between MINYEAR and MAXYEAR.
     * @type {number}
     */
    get year(): number;
    /**
     * Between 1 and 12.
     * @type {number}
     */
    get month(): number;
    /**
     * Between 1 and the number of days in the given month and year.
     * @type {number}
     */
    get day(): number;
    /**
     * Return a standard library Date object corresponding to this Date.
     * @param {boolean} utc If true, the value of getUTC***(), instead of
     *                      get***(), will correspond to this Date.
     * @returns {!stdDate}
     */
    toStdDate(utc?: boolean): stdDate;
    /**
     * Return a Date with the same value, except for those parameters given new
     * values by whichever keyword arguments are specified.
     * @param {Object} newValues The object consisting of new values.
     * @param {number} [newValues.year]
     * @param {number} [newValues.month]
     * @param {number} [newValues.day]
     * @returns {!Date}
     */
    replace({ year, month, day }: {
        year?: number | undefined;
        month?: number | undefined;
        day?: number | undefined;
    }): Date;
    /**
     * Return the proleptic Gregorian ordinal of the Date,
     * where January 1 of year 1 has ordinal 1.
     * For any Date object d, Date.fromordinal(d.toordinal()) == d.
     * @returns {number}
     */
    toOrdinal(): number;
    /**
     * Return the day of the week as an integer, where Monday is 0 and Sunday
     * is 6. For example, date(2002, 12, 4).weekday() == 2, a Wednesday.
     * @returns {number}
    */
    weekday(): number;
    /**
     * Return a string representing the date in ISO 8601 format, YYYY-MM-DD.
     * @returns {string}
     */
    isoFormat(): string;
    /**
     * Return a string representing the date, controlled by an explicit format
     * string. Format codes referring to hours, minutes or seconds will see 0
     * values.
     * @param {string} format The format string.
     * @returns {string}
     */
    strftime(format: string): string;
    /**
     * Same as isoFormat().
     * @returns {string}
     */
    toString(): string;
    /**
     * Same as toOrdinal().
     * @returns {number}
     */
    valueOf(): number;
}
/**
 * This is an abstract base class, meaning that this class should not be
 * instantiated directly. Define a subclass of tzinfo to capture information
 * about a particular time zone.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.tzinfo.
 */
export class TZInfo {
    /**
     * Return offset of local time from UTC, as a TimeDelta object that is
     * positive east of UTC. If local time is west of UTC, this should be
     * negative.
     * @param {?DateTime} dt The DateTime object.
     * @returns {?TimeDelta}
     */
    utcOffset(dt: DateTime | null): TimeDelta | null;
    /**
     * Return the daylight saving time (DST) adjustment, as a TimeDelta object
     * or null if DST information isn’t known.
     * @param {?DateTime} dt The DateTime object.
     * @returns {?TimeDelta}
     */
    dst(dt: DateTime | null): TimeDelta | null;
    /**
     * Return the time zone name corresponding to the datetime object dt, as a
     * string.
     * @param {?DateTime} dt The DateTime object.
     * @returns {?string}
     */
    tzName(dt: DateTime | null): string | null;
    /**
     * This is called from the default datetime.astimezone() implementation.
     * When called from that, dt.tzinfo is self, and dt’s date and time data are
     * to be viewed as expressing a UTC time. The purpose of fromutc() is to
     * adjust the date and time data, returning an equivalent datetime in self’s
     * local time.
     * @param {!DateTime} dt The DateTime object.
     * @returns {!DateTime}
     */
    fromUTC(dt: DateTime): DateTime;
}
/**
 * The TimeZone class is a subclass of TZInfo, each instance of which represents
 * a timezone defined by a fixed offset from UTC.
 * Objects of this class cannot be used to represent timezone information in the
 * locations where different offsets are used in different days of the year or
 * where historical changes have been made to civil time.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.timezone.
 */
export class TimeZone extends TZInfo {
    /**
     * The UTC timezone, new TimeZone(new TimeDelta(\{\})).
     * @type {!TimeZone}
     */
    static get utc(): TimeZone;
    /**
     *
     * @param {!TimeDelta} offset Represents the difference between the local
     *                            time and UTC. It must be strictly between
     *                            -TimeDelta(\{hours: 24\}) and
     *                            TimeDelta(\{hours: 24\}), otherwise
     *                            ValueDateTimeError is raised.
     * @param {?string} name If specified, it must be a string that will be used
     *                       as the value returned by the DateTime.tzname()
     *                       method.
     */
    constructor(offset: TimeDelta, name?: string | null);
    /**
     * @private
     * @readonly
     */
    private readonly _offset;
    /**
     * @private
     * @readonly
     */
    private readonly _name;
    /**
     * Return the fixed value specified when the TimeZone instance is
     * constructed.
     * @param {?DateTime} dt This argument is ignored.
     * @returns {!TimeDelta}
     */
    utcOffset(dt: DateTime | null): TimeDelta;
    /**
     * Return the fixed value specified when the timezone instance is
     * constructed.
     * If name is not provided in the constructor, the name returned by
     * tzName(dt) is generated from the value of the offset as follows.
     * If offset is TimeDelta(\{\}), the name is “UTC”, otherwise it is a string
     * in the format UTC±HH:MM, where ± is the sign of offset, HH and MM are two
     * digits of offset.hours and offset.minutes respectively.
     * @param {?DateTime} dt This argument is ignored.
     * @returns {string}
     */
    tzName(dt: DateTime | null): string;
    /**
     * Always returns null.
     * @param {?DateTime} dt This argument is ignored.
     * @returns {null}
     */
    dst(dt: DateTime | null): null;
}
/**
 * An instance of a class which is a subclass of TZInfo representing local
 * timezone of execution environment.
 */
export const LOCALTZINFO: LocalTZInfo;
/**
 * A time object represents a (local) time of day, independent of any particular
 * day, and subject to adjustment via a tzinfo object.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.time.
 */
export class Time {
    /**
     * Return a Time corresponding to a dateString given in the format
     * `HH[:MM[:SS[.fff[fff]]]][Z|((+|-)HH[:MM[:SS[.fff[fff]]]])]` or
     * `HH[MM[SS[.fff[fff]]]][Z|((+|-)HH[MM[SS[.fff[fff]]]])]`.

     * @param {string} timeString The time string.
     * @returns {!Time}
     */
    static fromISOFormat(timeString: string): Time;
    /**
     * The earliest representable time, new Time(0, 0, 0, 0).
     * @type {!Time}
     */
    static get min(): Time;
    /**
     * The latest representable time, new Time(23, 59, 59, 999999).
     * @type {!Time}
     */
    static get max(): Time;
    /**
     * The smallest possible difference between non-equal time objects,
     * new TimeDelta(\{microseconds: 1\}).
     * @type {!TimeDelta}
     */
    static get resolution(): TimeDelta;
    /**
     * @param {number} hour Between 0 and 23.
     * @param {number} minute Between 0 and 59.
     * @param {number} second Between 0 and 59.
     * @param {number} microsecond Between 0 and 999999.
     * @param {?TZInfo} tzInfo The timezone information.
     * @param {number} fold 0 or 1.
     */
    constructor(hour?: number, minute?: number, second?: number, microsecond?: number, tzInfo?: TZInfo | null, fold?: number);
    /**
     * @private
     * @readonly
     */
    private readonly _hour;
    /**
     * @private
     * @readonly
     */
    private readonly _minute;
    /**
     * @private
     * @readonly
     */
    private readonly _second;
    /**
     * @private
     * @readonly
     */
    private readonly _microsecond;
    /**
     * @private
     * @readonly
     */
    private readonly _tzInfo;
    /**
     * @private
     * @readonly
     */
    private readonly _fold;
    /**
     * Between 0 and 23.
     * @type {number}
     */
    get hour(): number;
    /**
     * Between 0 and 59.
     * @type {number}
     */
    get minute(): number;
    /**
     * Between 0 and 59.
     * @type {number}
     */
    get second(): number;
    /**
     * Between 0 and 999999.
     * @type {number}
     */
    get microsecond(): number;
    /**
     * The object passed as the tzInfo argument to the Time constructor, or null
     * if none was passed.
     * @type {?TZInfo}
     */
    get tzInfo(): TZInfo | null;
    /**
     * 0 or 1. Used to disambiguate wall times during a repeated interval.
     * (A repeated interval occurs when clocks are rolled back at the end of
     * daylight saving time or when the UTC offset for the current zone is
     * decreased for political reasons.) The value 0 (1) represents the earlier
     * (later) of the two moments with the same wall time representation.
     * @type {number}
     */
    get fold(): number;
    /**
     * Return a time with the same value, except for those attributes given new
     * values by whichever keyword arguments are specified. Note that
     * \{tzinfo: null\} can be specified to create a naive time from an aware
     * time, without conversion of the time data.
     * @param {Object} newValues The object consisting of new values.
     * @param {number} [newValues.hour]
     * @param {number} [newValues.minute]
     * @param {number} [newValues.second]
     * @param {number} [newValues.microsecond]
     * @param {?TZInfo} [newValues.tzInfo]
     * @param {number} [newValues.fold]
     * @returns {!Time}
     */
    replace({ hour, minute, second, microsecond, tzInfo, fold }: {
        hour?: number | undefined;
        minute?: number | undefined;
        second?: number | undefined;
        microsecond?: number | undefined;
        tzInfo?: TZInfo | null | undefined;
        fold?: number | undefined;
    }): Time;
    /**
     * Return a string representing the time in ISO 8601 format.
     * @param {"auto"|"microseconds"|"milliseconds"|"seconds"|"minutes"|"hours"
     * } timeSpec Specifies the number of additional components of the time to
     *            include.
     * @returns {string}
     */
    isoFormat(timeSpec?: "auto" | "microseconds" | "milliseconds" | "seconds" | "minutes" | "hours"): string;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.utcOffset(null).
     * @returns {?TimeDelta}
     */
    utcOffset(): TimeDelta | null;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.dst(null).
     * @returns {?TimeDelta}
     */
    dst(): TimeDelta | null;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.tzName(null).
     * @returns {?string}
     */
    tzName(): string | null;
    /**
     * Return a string representing the time, controlled by an explicit format
     * string.
     * @param {string} format The format string.
     * @returns {string}
     */
    strftime(format: string): string;
    /**
     * Same as isoFormat().
     * @returns {string}
     */
    toString(): string;
}
/**
 * A DateTime object is a single object containing all the information from a
 * Date object and a Time object.
 * Javascript version of
 * https://docs.python.org/3/library/datetime.html#datetime.datetime.
 */
export class DateTime extends Date {
    /**
     * Return a DateTime corresponding to the given standard library Date object.
     * @param {!stdDate} d The standard library Date object.
     * @param {boolean} utc If true, use getUTC***() instead of get***()
     *                      to construct DateTime.
     * @returns {!DateTime}
     */
    static fromStdDate(d: stdDate, utc?: boolean): DateTime;
    /**
     * Return the current local date and time, with tzInfo null.
     * @returns {!DateTime}
     */
    static today(): DateTime;
    /**
     * Return the current date and time.
     * @param {?TZInfo} tz If specified, the current date and time are converted
     *                     to tz's time zone, else same as today().
     * @returns {!DateTime}
     */
    static now(tz?: TZInfo | null): DateTime;
    /**
     * Return the current UTC date and time, with tzInfo null.
     * @returns {!DateTime}
     */
    static utcNow(): DateTime;
    /**
     * Return the local date and time corresponding to the POSIX timestamp.
     * @param {number} timeStamp The POSIX timestamp.
     * @param {?TZInfo} tz If null, the timestamp is converted to the platform's
     *                     local date and time, and the returned DateTime object
     *                     is naive. If not null, the timestamp is converted to
     *                     tz's time zone.
     * @returns {!DateTime}
     */
    static fromTimeStamp(timeStamp: number, tz?: TZInfo | null): DateTime;
    /**
     * Return the UTC date and time corresponding to the POSIX timestamp, with
     * tzInfo null. (The resulting object is naive.)
     * @param {number} timeStamp The POSIX timestamp.
     * @returns {!DateTime}
     */
    static utcFromTimeStamp(timeStamp: number): DateTime;
    /**
     * Return a new DateTime object whose date components are equal to the given
     * Date object’s, and whose time components are equal to the given Time
     * object’s. If the tzInfo argument is provided, its value is used to set
     * the tzInfo attribute of the result, otherwise the tzInfo attribute of the
     * time argument is used.
     * @param {!Date} date The Date object.
     * @param {!Time} time The Time object.
     * @param {?TZInfo} [tzInfo] The TZInfo object.
     * @returns {!DateTime}
     */
    static combine(date: Date, time: Time, tzInfo?: TZInfo | null | undefined): DateTime;
    /**
     * Return a DateTime corresponding to a dateString in one of the formats
     * emitted by Date.isoFormat() and DateTime.isoFormat().
     * @param {string} dateString The date string.
     * @returns {!DateTime}
     */
    static fromISOFormat(dateString: string): DateTime;
    /**
     * The earliest representable DateTime, new DateTime(MINYEAR, 1, 1).
     * @type {!DateTime}
     */
    static get min(): DateTime;
    /**
     * The latest representable DateTime, new DateTime(MAXYEAR, 12, 31, 23, 59,
     * 59, 999999).
     * @type {!DateTime}
     */
    static get max(): DateTime;
    /**
     * @param {number} year Between MINYEAR and MAXYEAR.
     * @param {number} month Between 1 and 12.
     * @param {number} day Between 1 and the number of days in the given month
     *                     and year.
     * @param {number} hour Between 0 and 23.
     * @param {number} minute Between 0 and 59.
     * @param {number} second Between 0 and 59.
     * @param {number} microsecond Between 0 and 999999.
     * @param {?TZInfo} tzInfo The timezone information.
     * @param {number} fold 0 or 1.
     */
    constructor(year: number, month: number, day: number, hour?: number, minute?: number, second?: number, microsecond?: number, tzInfo?: TZInfo | null, fold?: number);
    /**
     * @private
     * @readonly
     */
    private readonly _hour;
    /**
     * @private
     * @readonly
     */
    private readonly _minute;
    /**
     * @private
     * @readonly
     */
    private readonly _second;
    /**
     * @private
     * @readonly
     */
    private readonly _microsecond;
    /**
     * @private
     * @readonly
     */
    private readonly _tzInfo;
    /**
     * @private
     * @readonly
     */
    private readonly _fold;
    /**
     * Between 0 and 23.
     * @type {number}
     */
    get hour(): number;
    /**
     * Between 0 and 59.
     * @type {number}
     */
    get minute(): number;
    /**
     * Between 0 and 59.
     * @type {number}
     */
    get second(): number;
    /**
     * Between 0 and 999999.
     * @type {number}
     */
    get microsecond(): number;
    /**
     * The object passed as the tzInfo argument to the Time constructor, or null
     * if none was passed.
     * @type {?TZInfo}
     */
    get tzInfo(): TZInfo | null;
    /**
     * 0 or 1. Used to disambiguate wall times during a repeated interval.
     * (A repeated interval occurs when clocks are rolled back at the end of
     * daylight saving time or when the UTC offset for the current zone is
     * decreased for political reasons.) The value 0 (1) represents the earlier
     * (later) of the two moments with the same wall time representation.
     * @type {number}
     */
    get fold(): number;
    /**
     * Return Date object with same year, month and day.
     * @returns {!Date}
     */
    date(): Date;
    /**
     * Return Time object with same hour, minute, second, microsecond and fold.
     * tzInfo is null.
     * @returns {!Time}
     */
    time(): Time;
    /**
     * Return Time object with same hour, minute, second, microsecond, fold, and
     * tzInfo attributes.
     * @returns {!Time}
     */
    timetz(): Time;
    /**
     * Return a DateTime with the same attributes, except for those attributes
     * given new values by whichever keyword arguments are specified. Note that
     * \{tzInfo: null\} can be specified to create a naive DateTime from an
     * aware DateTime with no conversion of date and time data.
     * @param {Object} newValues The object consisting of new values.
     * @param {number} [newValues.year]
     * @param {number} [newValues.month]
     * @param {number} [newValues.day]
     * @param {number} [newValues.hour]
     * @param {number} [newValues.minute]
     * @param {number} [newValues.second]
     * @param {number} [newValues.microsecond]
     * @param {?TZInfo} [newValues.tzInfo]
     * @param {number} [newValues.fold]
     * @returns {!DateTime}
     */
    replace({ year, month, day, hour, minute, second, microsecond, tzInfo, fold, }: {
        year?: number | undefined;
        month?: number | undefined;
        day?: number | undefined;
        hour?: number | undefined;
        minute?: number | undefined;
        second?: number | undefined;
        microsecond?: number | undefined;
        tzInfo?: TZInfo | null | undefined;
        fold?: number | undefined;
    }): DateTime;
    /**
     * Return a DateTime object with new tzInfo attribute tz, adjusting the date
     * and time data so the result is the same UTC time as self, but in tz’s
     * local time.
     * If self is naive, it is presumed to represent time in the system timezone.
     * @param {?TZInfo} tz The target timezone. If null, the system local
     *                     timezone is assumed for the target timezone.
     * @returns {!DateTime}
     */
    asTimeZone(tz?: TZInfo | null): DateTime;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.utcOffset(this).
     * @returns {?TimeDelta}
     */
    utcOffset(): TimeDelta | null;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.dst(this).
     * @returns {?TimeDelta}
     */
    dst(): TimeDelta | null;
    /**
     * If tzInfo is null, returns null, else returns this.tzInfo.tzName(this).
     * @returns {?string}
     */
    tzName(): string | null;
    /**
     * Return POSIX timestamp corresponding to the DateTime instance.
     * @returns {number}
     */
    timeStamp(): number;
    /**
     * Return a string representing the date and time in ISO 8601 format.
     * @param {string} sep One-character separator placed between the date and
     *                     time portions of the result.
     * @param {"auto"|"microseconds"|"milliseconds"|"seconds"|"minutes"|"hours"
     * } timespec The number of additional components of the time to
     *            include.
     * @returns {string}
     */
    isoFormat(sep?: string, timespec?: "auto" | "microseconds" | "milliseconds" | "seconds" | "minutes" | "hours"): string;
}
export type stdDate = globalThis.Date;
/** @typedef {globalThis.Date} stdDate */
declare const stdDate: DateConstructor;
/**
 * A subclass of TZInfo representing local timezone of execution environment.
 */
declare class LocalTZInfo extends TZInfo {
    /**
     * @private
     * @readonly
     */
    private readonly _stdOffset;
    /**
     * Return offset of local time from UTC, as a TimeDelta object that is
     * positive east of UTC. If local time is west of UTC, this is negative.
     * @param {?DateTime} dt The DateTime object.
     * @returns {!TimeDelta}
     */
    utcOffset(dt: DateTime | null): TimeDelta;
    /**
     * Return the daylight saving time (DST) adjustment as a TimeDelta object.
     * @param {?DateTime} dt The DateTime object.
     * @returns {!TimeDelta}
     */
    dst(dt: DateTime | null): TimeDelta;
    /**
     * Return the time zone name corresponding to the datetime object dt, as a
     * string.
     * @param {?DateTime} dt The DateTime object.
     * @returns {string}
     */
    tzName(dt: DateTime | null): string;
}
export {};
