import "./chunk-76J2PTFD.js";

// node_modules/@silane/datetime/src/errors.js
var DateTimeError = class extends Error {
  /**
   * @param {string} message Some error message.
   */
  constructor(message) {
    super(message);
    Object.defineProperty(this, "name", {
      configurable: true,
      enumerable: false,
      value: this.constructor.name,
      writable: true
    });
    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, this.constructor);
    }
  }
};
var NotImplementedDateTimeError = class extends DateTimeError {
  constructor() {
    super("Not implemented.");
  }
};
var TypeDateTimeError = class extends DateTimeError {
  constructor(message = "Type error.") {
    super(message);
  }
};
var ValueDateTimeError = class extends DateTimeError {
  constructor(message = "Value error.") {
    super(message);
  }
};

// node_modules/@silane/datetime/src/datetime.js
var MINYEAR = 1;
var MAXYEAR = 9999;
var stdDate = globalThis.Date;
function safeStdDateUTC(year, month, day, hour, minute, second, millisecond) {
  const d = new stdDate(2e3, 0, 1);
  d.setUTCFullYear(year, month - 1, day);
  d.setUTCHours(hour, minute, second, millisecond);
  return d;
}
var daysPerMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
var leapedDaysPerMonth = [31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
var totalDaysPerMonth = function() {
  let sum = 0;
  const ret = daysPerMonth.map((x) => sum += x);
  ret.unshift(0);
  return ret;
}();
var totalLeapedDaysPerMonth = function() {
  let sum = 0;
  const ret = leapedDaysPerMonth.map((x) => sum += x);
  ret.unshift(0);
  return ret;
}();
function trySetProperty(obj, prop, value) {
  try {
    obj[prop] = value;
  } catch (e) {
    if (e instanceof TypeError) {
      return false;
    }
    throw e;
  }
  return Object.is(obj[prop], value);
}
function divmod(a, b) {
  const quotient = Math.floor(a / b);
  return [quotient, a - quotient * b];
}
function zeroPad(integer, length) {
  return integer.toString().padStart(length, "0");
}
function toOffsetString(timeDelta) {
  let offset = timeDelta;
  const minus = offset.days < 0;
  if (minus) {
    offset = neg(offset);
  }
  const seconds = offset.seconds % 60;
  const totalMinutes = Math.floor(offset.seconds / 60);
  const minutes = zeroPad(totalMinutes % 60, 2);
  const hours = zeroPad(Math.floor(totalMinutes / 60), 2);
  let ret = `${minus ? "-" : "+"}${hours}:${minutes}`;
  if (offset.microseconds) {
    ret += `:${zeroPad(seconds, 2)}.${zeroPad(offset.microseconds, 6)}`;
  } else if (seconds) {
    ret += `:${zeroPad(seconds, 2)}`;
  }
  return ret;
}
function isLeapYear(year) {
  if (year % 4 !== 0)
    return false;
  if (year % 100 === 0 && year % 400 !== 0)
    return false;
  return true;
}
function strftime(dt, format) {
  const a = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];
  const A = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
    "Sunday"
  ];
  const b = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec"
  ];
  const B = [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December"
  ];
  let ret = "";
  for (let i = 0; i < format.length; ++i) {
    if (format[i] !== "%" || i + 1 >= format.length) {
      ret += format[i];
      continue;
    }
    let s;
    switch (format[i + 1]) {
      case "a":
        s = a[dt.weekday()];
        break;
      case "A":
        s = A[dt.weekday()];
        break;
      case "w":
        s = ((dt.weekday() + 1) % 7).toString();
        break;
      case "d":
        s = zeroPad(dt.day, 2);
        break;
      case "b":
        s = b[dt.month - 1];
        break;
      case "B":
        s = B[dt.month - 1];
        break;
      case "m":
        s = zeroPad(dt.month, 2);
        break;
      case "y":
        s = zeroPad(dt.year % 100, 2);
        break;
      case "Y":
        s = zeroPad(dt.year, 4);
        break;
      case "H":
        s = zeroPad(dt.hour, 2);
        break;
      case "I":
        s = zeroPad(dt.hour % 12, 2);
        break;
      case "p":
        s = dt.hour < 12 ? "AM" : "PM";
        break;
      case "M":
        s = zeroPad(dt.minute, 2);
        break;
      case "S":
        s = zeroPad(dt.second, 2);
        break;
      case "f":
        s = zeroPad(dt.microsecond, 6);
        break;
      case "z":
        const offset = dt.utcOffset();
        if (offset == null)
          s = "";
        else
          s = toOffsetString(offset).replace(":", "");
        break;
      case "Z":
        const tzName = dt.tzName();
        if (tzName == null)
          s = "";
        else
          s = tzName;
        break;
      case "%":
        s = "%";
        break;
    }
    ret += s;
    ++i;
  }
  return ret;
}
var timeDeltaMin = null;
var timeDeltaMax = null;
var timeDeltaResolution = null;
var TimeDelta = class _TimeDelta {
  /**
   * @param {Object} duration An object consisting of duration values.
   * @param {number} [duration.days]
   * @param {number} [duration.seconds]
   * @param {number} [duration.microseconds]
   * @param {number} [duration.milliseconds]
   * @param {number} [duration.minutes]
   * @param {number} [duration.hours]
   * @param {number} [duration.weeks]
   */
  constructor({
    days = 0,
    seconds = 0,
    microseconds = 0,
    milliseconds = 0,
    minutes = 0,
    hours = 0,
    weeks = 0
  } = {}) {
    microseconds += milliseconds * 1e3;
    seconds += minutes * 60;
    seconds += hours * 3600;
    days += weeks * 7;
    let frac;
    [days, frac] = divmod(days, 1);
    seconds += frac * 3600 * 24;
    [seconds, frac] = divmod(seconds, 1);
    microseconds += frac * 1e3 ** 2;
    microseconds = Math.round(microseconds);
    let div, mod;
    [div, mod] = divmod(microseconds, 1e3 ** 2);
    microseconds = mod;
    seconds += div;
    [div, mod] = divmod(seconds, 3600 * 24);
    seconds = mod;
    days += div;
    if (!(-999999999 <= days && days <= 999999999)) {
      throw new ValueDateTimeError(
        'Cannot handle duration greater than "TimeDelta.max" or lesser than "TimeDelta.min".'
      );
    }
    this._days = days;
    this._seconds = seconds;
    this._microseconds = microseconds;
  }
  /**
   * Between -999999999 and 999999999 inclusive.
   * @type {number}
   */
  get days() {
    return this._days;
  }
  /**
   * Between 0 and 86399 inclusive
   * @type {number}
   */
  get seconds() {
    return this._seconds;
  }
  /**
   * Between 0 and 999999 inclusive
   * @type {number}
   */
  get microseconds() {
    return this._microseconds;
  }
  /**
   * Return the total number of seconds contained in the duration.
   * @returns {number}
   */
  totalSeconds() {
    if (this["_totalSeconds"] != null)
      return this["_totalSeconds"];
    const ret = this.days * 3600 * 24 + this.seconds + this.microseconds / 1e6;
    trySetProperty(this, "_totalSeconds", ret);
    return ret;
  }
  /**
   * Return the human-readable string respresentation.
   * @returns {string}
   */
  toString() {
    if (this["_string"] != null)
      return this["_string"];
    let ret = "";
    if (this.days) {
      ret += `${this.days} day(s), `;
    }
    const totalMinutes = Math.floor(this.seconds / 60);
    ret += `${Math.floor(totalMinutes / 60)}:${zeroPad(totalMinutes % 60, 2)}:${zeroPad(this.seconds % 60, 2)}`;
    if (this.microseconds) {
      ret += `.${zeroPad(this.microseconds, 6)}`;
    }
    trySetProperty(this, "_string", ret);
    return ret;
  }
  /**
   * Same as totalSeconds().
   * @returns {number}
   */
  valueOf() {
    return this.totalSeconds();
  }
  /**
   * The most negative timedelta object, new TimeDelta(\{days: -999999999\}).
   * @type {!TimeDelta}
   */
  static get min() {
    if (!timeDeltaMin) {
      timeDeltaMin = new _TimeDelta({ days: -999999999 });
    }
    return timeDeltaMin;
  }
  /**
   * The most positive timedelta object, new TimeDelta(\{days: 999999999,
   * hours: 23, minutes: 59, seconds: 59, microseconds: 999999\}).
   * @type {!TimeDelta}
   */
  static get max() {
    if (!timeDeltaMax) {
      timeDeltaMax = new _TimeDelta({
        days: 999999999,
        hours: 23,
        minutes: 59,
        seconds: 59,
        microseconds: 999999
      });
    }
    return timeDeltaMax;
  }
  /**
   * The smallest possible difference between non-equal timedelta objects,
   * new TimeDelta(\{microseconds: 1\}).
   * @type {!TimeDelta}
  */
  static get resolution() {
    if (!timeDeltaResolution) {
      timeDeltaResolution = new _TimeDelta({ microseconds: 1 });
    }
    return timeDeltaResolution;
  }
};
var dateMin = null;
var dateMax = null;
var dateResolution = null;
var Date = class _Date {
  /**
   * @param {number} year Between MINYEAR and MAXYEAR.
   * @param {number} month Between 1 and 12.
   * @param {number} day Between 1 and the number of days in the given month
   *                     and year.
   */
  constructor(year, month, day) {
    if (!(MINYEAR <= year && year <= MAXYEAR))
      throw new ValueDateTimeError(
        '"year" must be between "MINYEAR" and "MAXYEAR".'
      );
    if (!(1 <= month && month <= 12))
      throw new ValueDateTimeError('"month" must be between 1 and 12.');
    if (!(1 <= day && day <= (isLeapYear(year) ? leapedDaysPerMonth[month - 1] : daysPerMonth[month - 1])))
      throw new ValueDateTimeError("Invalid day for the year and month.");
    this._year = year;
    this._month = month;
    this._day = day;
  }
  /**
   * Between MINYEAR and MAXYEAR.
   * @type {number}
   */
  get year() {
    return this._year;
  }
  /**
   * Between 1 and 12.
   * @type {number}
   */
  get month() {
    return this._month;
  }
  /**
   * Between 1 and the number of days in the given month and year.
   * @type {number}
   */
  get day() {
    return this._day;
  }
  /**
   * Return the Date corresponding to the given standard library Date object.
   * @param {!stdDate} d The standard library Date object.
   * @param {boolean} utc If true, use getUTC***() instead of get***()
   *                      to construct Date.
   * @returns {!Date}
   */
  static fromStdDate(d, utc = false) {
    if (!utc)
      return new _Date(
        d.getFullYear(),
        d.getMonth() + 1,
        d.getDate()
      );
    else
      return new _Date(
        d.getUTCFullYear(),
        d.getUTCMonth() + 1,
        d.getUTCDate()
      );
  }
  /**
   * Return the current local date.
   * @returns {!Date}
   */
  static today() {
    const today = new stdDate();
    return _Date.fromStdDate(today);
  }
  /**
   * Return the Date corresponding to the proleptic Gregorian ordinal, where
   * January 1 of year 1 has ordinal 1.
   * @param {number} ordinal The proleptic Gregorian ordinal.
   * @returns {!Date}
   */
  static fromOrdinal(ordinal) {
    let q, r;
    let year = 1, month = 1, day = 1;
    [q, r] = divmod(ordinal - 1, 365 * 303 + 366 * 97);
    year += q * 400;
    [q, r] = divmod(r, 365 * 76 + 366 * 24);
    year += q * 100;
    [q, r] = divmod(r, 365 * 3 + 366 * 1);
    year += q * 4;
    [q, r] = divmod(r, 365);
    if (q <= 2) {
      year += q;
      for (month = 1; month <= 12 && r >= totalDaysPerMonth[month]; ++month)
        ;
      day += r - totalDaysPerMonth[month - 1];
    } else {
      year += 3;
      if (q === 4)
        r += 365;
      for (month = 1; month <= 12 && r >= totalLeapedDaysPerMonth[month]; ++month)
        ;
      day += r - totalLeapedDaysPerMonth[month - 1];
    }
    return new _Date(year, month, day);
  }
  /**
   * Return a Date corresponding to a dateString given in the format
   * `YYYY-MM-DD` or `YYYYMMDD`.
   * @param {string} dateString The date string.
   * @returns {!Date}
   */
  static fromISOFormat(dateString) {
    let match = /^(\d\d\d\d)-(\d\d)-(\d\d)$/.exec(dateString) || /^(\d\d\d\d)(\d\d)(\d\d)$/.exec(dateString);
    if (match == null) {
      throw new ValueDateTimeError("Invalid format.");
    }
    const [year, month, day] = match.slice(1).map(Number);
    return new _Date(year, month, day);
  }
  /**
   * Return a standard library Date object corresponding to this Date.
   * @param {boolean} utc If true, the value of getUTC***(), instead of
   *                      get***(), will correspond to this Date.
   * @returns {!stdDate}
   */
  toStdDate(utc = false) {
    let ret;
    if (!utc) {
      ret = new stdDate(this.year, this.month - 1, this.day);
      ret.setFullYear(this.year);
    } else {
      ret = new stdDate(stdDate.UTC(this.year, this.month - 1, this.day));
      ret.setUTCFullYear(this.year);
    }
    return ret;
  }
  /**
   * Return a Date with the same value, except for those parameters given new
   * values by whichever keyword arguments are specified.
   * @param {Object} newValues The object consisting of new values.
   * @param {number} [newValues.year]
   * @param {number} [newValues.month]
   * @param {number} [newValues.day]
   * @returns {!Date}
   */
  replace({ year = this.year, month = this.month, day = this.day }) {
    return new _Date(year, month, day);
  }
  /**
   * Return the proleptic Gregorian ordinal of the Date,
   * where January 1 of year 1 has ordinal 1.
   * For any Date object d, Date.fromordinal(d.toordinal()) == d.
   * @returns {number}
   */
  toOrdinal() {
    if (this["_ordinal"] != null)
      return this["_ordinal"];
    let totalDays = 0;
    const lastYear = this.year - 1;
    const nLeapYear = Math.floor(lastYear / 4) - Math.floor(lastYear / 100) + Math.floor(lastYear / 400);
    totalDays += nLeapYear * 366 + (lastYear - nLeapYear) * 365;
    if (isLeapYear(this.year)) {
      totalDays += totalLeapedDaysPerMonth[this.month - 1];
    } else {
      totalDays += totalDaysPerMonth[this.month - 1];
    }
    totalDays += this.day;
    trySetProperty(this, "_ordinal", totalDays);
    return totalDays;
  }
  /**
   * Return the day of the week as an integer, where Monday is 0 and Sunday
   * is 6. For example, date(2002, 12, 4).weekday() == 2, a Wednesday.
   * @returns {number}
  */
  weekday() {
    return (this.toStdDate().getDay() + 6) % 7;
  }
  /**
   * Return a string representing the date in ISO 8601 format, YYYY-MM-DD.
   * @returns {string}
   */
  isoFormat() {
    if (this["_isoFormat"] != null)
      return this["_isoFormat"];
    const ret = `${zeroPad(this.year, 4)}-${zeroPad(this.month, 2)}-${zeroPad(this.day, 2)}`;
    trySetProperty(this, "_isoFormat", ret);
    return ret;
  }
  /**
   * Return a string representing the date, controlled by an explicit format
   * string. Format codes referring to hours, minutes or seconds will see 0
   * values.
   * @param {string} format The format string.
   * @returns {string}
   */
  strftime(format) {
    const dt = DateTime.combine(this, new Time());
    return strftime(dt, format);
  }
  /**
   * Same as isoFormat().
   * @returns {string}
   */
  toString() {
    return this.isoFormat();
  }
  /**
   * Same as toOrdinal().
   * @returns {number}
   */
  valueOf() {
    return this.toOrdinal();
  }
  /**
   * The earliest representable date, new Date(MINYEAR, 1, 1).
   * @type {!Date}
   */
  static get min() {
    if (!dateMin) {
      dateMin = new _Date(MINYEAR, 1, 1);
    }
    return dateMin;
  }
  /**
   * The latest representable date, new Date(MAXYEAR, 12, 31).
   * @type {!Date}
   */
  static get max() {
    if (!dateMax) {
      dateMax = new _Date(MAXYEAR, 12, 31);
    }
    return dateMax;
  }
  /**
   * The smallest possible difference between non-equal date objects,
   * new TimeDelta(\{days: 1\}).
   * @type {!TimeDelta}
   */
  static get resolution() {
    if (!dateResolution) {
      dateResolution = new TimeDelta({ days: 1 });
    }
    return dateResolution;
  }
};
var TZInfo = class {
  /**
   * Return offset of local time from UTC, as a TimeDelta object that is
   * positive east of UTC. If local time is west of UTC, this should be
   * negative.
   * @param {?DateTime} dt The DateTime object.
   * @returns {?TimeDelta}
   */
  utcOffset(dt) {
    throw new NotImplementedDateTimeError();
  }
  /**
   * Return the daylight saving time (DST) adjustment, as a TimeDelta object
   * or null if DST information isn’t known.
   * @param {?DateTime} dt The DateTime object.
   * @returns {?TimeDelta}
   */
  dst(dt) {
    throw new NotImplementedDateTimeError();
  }
  /**
   * Return the time zone name corresponding to the datetime object dt, as a
   * string.
   * @param {?DateTime} dt The DateTime object.
   * @returns {?string}
   */
  tzName(dt) {
    throw new NotImplementedDateTimeError();
  }
  /**
   * This is called from the default datetime.astimezone() implementation.
   * When called from that, dt.tzinfo is self, and dt’s date and time data are
   * to be viewed as expressing a UTC time. The purpose of fromutc() is to
   * adjust the date and time data, returning an equivalent datetime in self’s
   * local time.
   * @param {!DateTime} dt The DateTime object.
   * @returns {!DateTime}
   */
  fromUTC(dt) {
    if (dt.tzInfo !== this) {
      throw new ValueDateTimeError(
        '"dt.tzInfo" must be same instance as "this".'
      );
    }
    let dtoff = dt.utcOffset();
    let dtdst = dt.dst();
    if (dtoff == null || dtdst == null) {
      throw new ValueDateTimeError(
        '"dt.utcOffset()" and "dt.dst()" must not return null.'
      );
    }
    const delta = sub(dtoff, dtdst);
    if (cmp(delta, new TimeDelta()) !== 0) {
      dt = add(dt, delta);
      dtdst = dt.dst();
    }
    if (dtdst == null)
      return dt;
    else
      return add(dt, dtdst);
  }
};
var timeZoneUTC = null;
var TimeZone = class _TimeZone extends TZInfo {
  /**
   *
   * @param {!TimeDelta} offset Represents the difference between the local
   *                            time and UTC. It must be strictly between
   *                            -TimeDelta(\{hours: 24\}) and
   *                            TimeDelta(\{hours: 24\}), otherwise
   *                            ValueDateTimeError is raised.
   * @param {?string} name If specified, it must be a string that will be used
   *                       as the value returned by the DateTime.tzname()
   *                       method.
   */
  constructor(offset, name = null) {
    super();
    if (!(cmp(new TimeDelta({ hours: -24 }), offset) < 0 && cmp(offset, new TimeDelta({ hours: 24 })) < 0))
      throw new ValueDateTimeError(
        '"offset" must be "TimeDelta({hours: -24}) < offset < TimeDelta({hours: 24})".'
      );
    if (name == null) {
      name = "UTC";
      if (cmp(offset, new TimeDelta()) != 0) {
        name += toOffsetString(offset);
      }
    }
    this._offset = offset;
    this._name = name;
  }
  /**
   * Return the fixed value specified when the TimeZone instance is
   * constructed.
   * @param {?DateTime} dt This argument is ignored.
   * @returns {!TimeDelta}
   */
  utcOffset(dt) {
    return this._offset;
  }
  /**
   * Return the fixed value specified when the timezone instance is
   * constructed.
   * If name is not provided in the constructor, the name returned by
   * tzName(dt) is generated from the value of the offset as follows.
   * If offset is TimeDelta(\{\}), the name is “UTC”, otherwise it is a string
   * in the format UTC±HH:MM, where ± is the sign of offset, HH and MM are two
   * digits of offset.hours and offset.minutes respectively.
   * @param {?DateTime} dt This argument is ignored.
   * @returns {string}
   */
  tzName(dt) {
    return this._name;
  }
  /**
   * Always returns null.
   * @param {?DateTime} dt This argument is ignored.
   * @returns {null}
   */
  dst(dt) {
    return null;
  }
  /**
   * Return add(dt, offset). The dt argument must be an aware datetime
   * instance, with tzInfo set to this.
   * @param {!DateTime} dt The DateTime object.
   * @returns {!DateTime}
   */
  fromUTC(dt) {
    if (dt.tzInfo !== this) {
      throw new ValueDateTimeError(
        '"dt.tzInfo" must be same instance as "this".'
      );
    }
    return add(dt, this._offset);
  }
  /**
   * The UTC timezone, new TimeZone(new TimeDelta(\{\})).
   * @type {!TimeZone}
   */
  static get utc() {
    if (!timeZoneUTC) {
      timeZoneUTC = new _TimeZone(new TimeDelta({}));
    }
    return timeZoneUTC;
  }
};
var LocalTZInfo = class extends TZInfo {
  constructor() {
    super();
    const stdOffset = -new stdDate(2e3, 0, 1).getTimezoneOffset();
    this._stdOffset = new TimeDelta({ minutes: stdOffset });
  }
  /**
   * Return offset of local time from UTC, as a TimeDelta object that is
   * positive east of UTC. If local time is west of UTC, this is negative.
   * @param {?DateTime} dt The DateTime object.
   * @returns {!TimeDelta}
   */
  utcOffset(dt) {
    if (dt == null)
      return this._stdOffset;
    const offset = -dt.toStdDate(false).getTimezoneOffset();
    return new TimeDelta({ minutes: offset });
  }
  /**
   * Return the daylight saving time (DST) adjustment as a TimeDelta object.
   * @param {?DateTime} dt The DateTime object.
   * @returns {!TimeDelta}
   */
  dst(dt) {
    if (dt == null)
      return new TimeDelta();
    const offsetMinutes = -dt.toStdDate(false).getTimezoneOffset();
    const offset = new TimeDelta({ minutes: offsetMinutes });
    return sub(offset, this._stdOffset);
  }
  /**
   * Return the time zone name corresponding to the datetime object dt, as a
   * string.
   * @param {?DateTime} dt The DateTime object.
   * @returns {string}
   */
  tzName(dt) {
    const offset = this.utcOffset(dt);
    return toOffsetString(offset);
  }
  /**
   * This is called from the default datetime.astimezone() implementation.
   * When called from that, dt.tzinfo is self, and dt’s date and time data are
   * to be viewed as expressing a UTC time. The purpose of fromutc() is to
   * adjust the date and time data, returning an equivalent datetime in self’s
   * local time.
   * @param {!DateTime} dt The DateTime object.
   * @returns {!DateTime}
   */
  fromUTC(dt) {
    if (dt.tzInfo !== this)
      throw new ValueDateTimeError(
        '"dt.tzInfo" must be same instance as "this".'
      );
    const local = DateTime.fromStdDate(dt.toStdDate(true), false).replace({
      microsecond: dt.microsecond,
      tzInfo: this,
      fold: 0
    });
    return local;
  }
};
var LOCALTZINFO = new LocalTZInfo();
var timeMin = null;
var timeMax = null;
var timeResolution = null;
var Time = class _Time {
  /**
   * @param {number} hour Between 0 and 23.
   * @param {number} minute Between 0 and 59.
   * @param {number} second Between 0 and 59.
   * @param {number} microsecond Between 0 and 999999.
   * @param {?TZInfo} tzInfo The timezone information.
   * @param {number} fold 0 or 1.
   */
  constructor(hour = 0, minute = 0, second = 0, microsecond = 0, tzInfo = null, fold = 0) {
    if (!(0 <= hour && hour <= 23))
      throw new ValueDateTimeError(
        '"hour" must be between 0 and 23.'
      );
    if (!(0 <= minute && minute <= 59))
      throw new ValueDateTimeError(
        '"minute" must be between 0 and 59.'
      );
    if (!(0 <= second && second <= 59))
      throw new ValueDateTimeError(
        '"second" must be between 0 and 59.'
      );
    if (!(0 <= microsecond && microsecond <= 999999))
      throw new ValueDateTimeError(
        '"microsecond" must be between 0 and 999999.'
      );
    if (!(fold === 0 || fold === 1))
      throw new ValueDateTimeError(
        '"fold" must be 0 or 1.'
      );
    this._hour = hour;
    this._minute = minute;
    this._second = second;
    this._microsecond = microsecond;
    this._tzInfo = tzInfo;
    this._fold = fold;
  }
  /**
   * Between 0 and 23.
   * @type {number}
   */
  get hour() {
    return this._hour;
  }
  /**
   * Between 0 and 59.
   * @type {number}
   */
  get minute() {
    return this._minute;
  }
  /**
   * Between 0 and 59.
   * @type {number}
   */
  get second() {
    return this._second;
  }
  /**
   * Between 0 and 999999.
   * @type {number}
   */
  get microsecond() {
    return this._microsecond;
  }
  /**
   * The object passed as the tzInfo argument to the Time constructor, or null
   * if none was passed.
   * @type {?TZInfo}
   */
  get tzInfo() {
    return this._tzInfo;
  }
  /**
   * 0 or 1. Used to disambiguate wall times during a repeated interval.
   * (A repeated interval occurs when clocks are rolled back at the end of
   * daylight saving time or when the UTC offset for the current zone is
   * decreased for political reasons.) The value 0 (1) represents the earlier
   * (later) of the two moments with the same wall time representation.
   * @type {number}
   */
  get fold() {
    return this._fold;
  }
  /**
       * Return a Time corresponding to a dateString given in the format
       * `HH[:MM[:SS[.fff[fff]]]][Z|((+|-)HH[:MM[:SS[.fff[fff]]]])]` or
       * `HH[MM[SS[.fff[fff]]]][Z|((+|-)HH[MM[SS[.fff[fff]]]])]`.
  
       * @param {string} timeString The time string.
       * @returns {!Time}
       */
  static fromISOFormat(timeString) {
    function parseTimeString(str) {
      const match = /^(\d\d)(?:\:(\d\d)(?:\:(\d\d)(?:\.(\d{3})(\d{3})?)?)?)?$/.exec(str) || /^(\d\d)(?:(\d\d)(?:(\d\d)(?:\.(\d{3})(\d{3})?)?)?)?$/.exec(str);
      if (match == null)
        return null;
      match.splice(0, 1);
      const ret = match.map((x) => x == null ? 0 : parseInt(x, 10));
      ret[3] = ret[3] * 1e3 + ret[4];
      ret.splice(4, 1);
      return ret;
    }
    let sepIdx = timeString.search(/[Z+-]/);
    if (sepIdx === -1)
      sepIdx = timeString.length;
    const timeStr = timeString.slice(0, sepIdx);
    const offsetStr = timeString.slice(sepIdx);
    const timeArray = parseTimeString(timeStr);
    if (timeArray == null)
      throw new ValueDateTimeError(
        "Invalid format."
      );
    let tzInfo = null;
    if (offsetStr === "Z") {
      tzInfo = new TimeZone(new TimeDelta({}));
    } else if (offsetStr !== "") {
      const offsetArray = parseTimeString(offsetStr.slice(1));
      if (offsetArray == null) {
        throw new ValueDateTimeError(
          "Invalid format."
        );
      }
      let offset = new TimeDelta({
        hours: offsetArray[0],
        minutes: offsetArray[1],
        seconds: offsetArray[2],
        microseconds: offsetArray[3]
      });
      if (offsetStr[0] === "-")
        offset = neg(offset);
      tzInfo = new TimeZone(offset);
    }
    return new _Time(
      timeArray[0],
      timeArray[1],
      timeArray[2],
      timeArray[3],
      tzInfo
    );
  }
  /**
   * Return a time with the same value, except for those attributes given new
   * values by whichever keyword arguments are specified. Note that
   * \{tzinfo: null\} can be specified to create a naive time from an aware
   * time, without conversion of the time data.
   * @param {Object} newValues The object consisting of new values.
   * @param {number} [newValues.hour]
   * @param {number} [newValues.minute]
   * @param {number} [newValues.second]
   * @param {number} [newValues.microsecond]
   * @param {?TZInfo} [newValues.tzInfo]
   * @param {number} [newValues.fold]
   * @returns {!Time}
   */
  replace({
    hour = this.hour,
    minute = this.minute,
    second = this.second,
    microsecond = this.microsecond,
    tzInfo = this.tzInfo,
    fold = this.fold
  }) {
    return new _Time(hour, minute, second, microsecond, tzInfo, fold);
  }
  /**
   * Return a string representing the time in ISO 8601 format.
   * @param {"auto"|"microseconds"|"milliseconds"|"seconds"|"minutes"|"hours"
   * } timeSpec Specifies the number of additional components of the time to
   *            include.
   * @returns {string}
   */
  isoFormat(timeSpec = "auto") {
    var _a;
    if (timeSpec === "auto") {
      timeSpec = this.microsecond ? "microseconds" : "seconds";
    }
    if (((_a = this["_isoFormat"]) == null ? void 0 : _a[timeSpec]) != null) {
      return this["_isoFormat"][timeSpec];
    }
    let ret = "";
    switch (timeSpec) {
      case "microseconds":
      case "milliseconds":
        if (timeSpec === "microseconds")
          ret = zeroPad(this.microsecond, 6) + ret;
        else
          ret = zeroPad(Math.floor(this.microsecond / 1e3), 3) + ret;
        ret = "." + ret;
      case "seconds":
        ret = ":" + zeroPad(this.second, 2) + ret;
      case "minutes":
        ret = ":" + zeroPad(this.minute, 2) + ret;
      case "hours":
        ret = zeroPad(this.hour, 2) + ret;
        break;
      default:
        throw new ValueDateTimeError(
          '"timeSpec" must be either "auto", "microseconds", "milliseconds", "seconds", "minutes" or "hours".'
        );
    }
    const offset = this.utcOffset();
    if (offset != null) {
      ret += toOffsetString(offset);
    }
    if (!this["_isoFormat"]) {
      trySetProperty(this, "_isoFormat", {});
    }
    if (this["_isoFormat"]) {
      trySetProperty(this["_isoFormat"], timeSpec, ret);
    }
    return ret;
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.utcOffset(null).
   * @returns {?TimeDelta}
   */
  utcOffset() {
    return this.tzInfo == null ? null : this.tzInfo.utcOffset(null);
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.dst(null).
   * @returns {?TimeDelta}
   */
  dst() {
    return this.tzInfo == null ? null : this.tzInfo.dst(null);
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.tzName(null).
   * @returns {?string}
   */
  tzName() {
    return this.tzInfo == null ? null : this.tzInfo.tzName(null);
  }
  /**
   * Return a string representing the time, controlled by an explicit format
   * string.
   * @param {string} format The format string.
   * @returns {string}
   */
  strftime(format) {
    const dt = DateTime.combine(new Date(1900, 1, 1), this);
    return strftime(dt, format);
  }
  /**
   * Same as isoFormat().
   * @returns {string}
   */
  toString() {
    return this.isoFormat();
  }
  /**
   * The earliest representable time, new Time(0, 0, 0, 0).
   * @type {!Time}
   */
  static get min() {
    if (!timeMin) {
      timeMin = new _Time(0, 0, 0, 0);
    }
    return timeMin;
  }
  /**
   * The latest representable time, new Time(23, 59, 59, 999999).
   * @type {!Time}
   */
  static get max() {
    if (!timeMax) {
      timeMax = new _Time(23, 59, 59, 999999);
    }
    return timeMax;
  }
  /**
   * The smallest possible difference between non-equal time objects,
   * new TimeDelta(\{microseconds: 1\}).
   * @type {!TimeDelta}
   */
  static get resolution() {
    if (!timeResolution) {
      timeResolution = new TimeDelta({ microseconds: 1 });
    }
    return timeResolution;
  }
};
var dateTimeMin = null;
var dateTimeMax = null;
var dateTimeResolution = null;
var DateTime = class _DateTime extends Date {
  /**
   * @param {number} year Between MINYEAR and MAXYEAR.
   * @param {number} month Between 1 and 12.
   * @param {number} day Between 1 and the number of days in the given month
   *                     and year.
   * @param {number} hour Between 0 and 23.
   * @param {number} minute Between 0 and 59.
   * @param {number} second Between 0 and 59.
   * @param {number} microsecond Between 0 and 999999.
   * @param {?TZInfo} tzInfo The timezone information.
   * @param {number} fold 0 or 1.
   */
  constructor(year, month, day, hour = 0, minute = 0, second = 0, microsecond = 0, tzInfo = null, fold = 0) {
    super(year, month, day);
    if (!(0 <= hour && hour <= 23))
      throw new ValueDateTimeError(
        '"hour" must be between 0 and 23.'
      );
    if (!(0 <= minute && minute <= 59))
      throw new ValueDateTimeError(
        '"minute" must be between 0 and 59.'
      );
    if (!(0 <= second && second <= 59))
      throw new ValueDateTimeError(
        '"second" must be between 0 and 59.'
      );
    if (!(0 <= microsecond && microsecond <= 999999))
      throw new ValueDateTimeError(
        '"microsecond" must be between 0 and 999999.'
      );
    if (!(fold === 0 || fold === 1))
      throw new ValueDateTimeError(
        '"fold" must be 0 or 1.'
      );
    this._hour = hour;
    this._minute = minute;
    this._second = second;
    this._microsecond = microsecond;
    this._tzInfo = tzInfo;
    this._fold = fold;
  }
  /**
   * Between 0 and 23.
   * @type {number}
   */
  get hour() {
    return this._hour;
  }
  /**
   * Between 0 and 59.
   * @type {number}
   */
  get minute() {
    return this._minute;
  }
  /**
   * Between 0 and 59.
   * @type {number}
   */
  get second() {
    return this._second;
  }
  /**
   * Between 0 and 999999.
   * @type {number}
   */
  get microsecond() {
    return this._microsecond;
  }
  /**
   * The object passed as the tzInfo argument to the Time constructor, or null
   * if none was passed.
   * @type {?TZInfo}
   */
  get tzInfo() {
    return this._tzInfo;
  }
  /**
   * 0 or 1. Used to disambiguate wall times during a repeated interval.
   * (A repeated interval occurs when clocks are rolled back at the end of
   * daylight saving time or when the UTC offset for the current zone is
   * decreased for political reasons.) The value 0 (1) represents the earlier
   * (later) of the two moments with the same wall time representation.
   * @type {number}
   */
  get fold() {
    return this._fold;
  }
  /**
   * Return a DateTime corresponding to the given standard library Date object.
   * @param {!stdDate} d The standard library Date object.
   * @param {boolean} utc If true, use getUTC***() instead of get***()
   *                      to construct DateTime.
   * @returns {!DateTime}
   */
  static fromStdDate(d, utc = false) {
    if (!utc)
      return new _DateTime(
        d.getFullYear(),
        d.getMonth() + 1,
        d.getDate(),
        d.getHours(),
        d.getMinutes(),
        d.getSeconds(),
        d.getMilliseconds() * 1e3
      );
    else
      return new _DateTime(
        d.getUTCFullYear(),
        d.getUTCMonth() + 1,
        d.getUTCDate(),
        d.getUTCHours(),
        d.getUTCMinutes(),
        d.getUTCSeconds(),
        d.getUTCMilliseconds() * 1e3
      );
  }
  /**
   * Return the current local date and time, with tzInfo null.
   * @returns {!DateTime}
   */
  static today() {
    return _DateTime.fromStdDate(new stdDate());
  }
  /**
   * Return the current date and time.
   * @param {?TZInfo} tz If specified, the current date and time are converted
   *                     to tz's time zone, else same as today().
   * @returns {!DateTime}
   */
  static now(tz = null) {
    if (tz == null)
      return _DateTime.today();
    return tz.fromUTC(_DateTime.utcNow().replace({ tzInfo: tz }));
  }
  /**
   * Return the current UTC date and time, with tzInfo null.
   * @returns {!DateTime}
   */
  static utcNow() {
    return _DateTime.fromStdDate(new stdDate(), true);
  }
  /**
   * Return the local date and time corresponding to the POSIX timestamp.
   * @param {number} timeStamp The POSIX timestamp.
   * @param {?TZInfo} tz If null, the timestamp is converted to the platform's
   *                     local date and time, and the returned DateTime object
   *                     is naive. If not null, the timestamp is converted to
   *                     tz's time zone.
   * @returns {!DateTime}
   */
  static fromTimeStamp(timeStamp, tz = null) {
    if (tz == null)
      return _DateTime.fromStdDate(new stdDate(timeStamp * 1e3));
    return tz.fromUTC(
      _DateTime.utcFromTimeStamp(timeStamp).replace({ tzInfo: tz })
    );
  }
  /**
   * Return the UTC date and time corresponding to the POSIX timestamp, with
   * tzInfo null. (The resulting object is naive.)
   * @param {number} timeStamp The POSIX timestamp.
   * @returns {!DateTime}
   */
  static utcFromTimeStamp(timeStamp) {
    return _DateTime.fromStdDate(new stdDate(timeStamp * 1e3), true);
  }
  /**
   * Return a new DateTime object whose date components are equal to the given
   * Date object’s, and whose time components are equal to the given Time
   * object’s. If the tzInfo argument is provided, its value is used to set
   * the tzInfo attribute of the result, otherwise the tzInfo attribute of the
   * time argument is used.
   * @param {!Date} date The Date object.
   * @param {!Time} time The Time object.
   * @param {?TZInfo} [tzInfo] The TZInfo object.
   * @returns {!DateTime}
   */
  static combine(date2, time2, tzInfo = void 0) {
    if (tzInfo === void 0)
      tzInfo = time2.tzInfo;
    return new _DateTime(
      date2.year,
      date2.month,
      date2.day,
      time2.hour,
      time2.minute,
      time2.second,
      time2.microsecond,
      tzInfo,
      time2.fold
    );
  }
  /**
   * Return a DateTime corresponding to a dateString in one of the formats
   * emitted by Date.isoFormat() and DateTime.isoFormat().
   * @param {string} dateString The date string.
   * @returns {!DateTime}
   */
  static fromISOFormat(dateString) {
    let sepIdx = dateString.search(/[^\d\-]/);
    if (sepIdx === -1)
      sepIdx = dateString.length;
    const dateStr = dateString.slice(0, sepIdx);
    const timeStr = dateString.slice(sepIdx + 1);
    return _DateTime.combine(
      Date.fromISOFormat(dateStr),
      timeStr ? Time.fromISOFormat(timeStr) : new Time()
    );
  }
  /**
   * Return a standard library Date object corresponding to this DateTime.
   * Since standard library Date object has only millisecond resolution, the
   * microsecond value is truncated.
   * @param {boolean} utc If true, the value of getUTC***(), instead of
   *                      get***(), will correspond to this Date.
   * @returns {!stdDate}
   */
  toStdDate(utc = false) {
    let ret;
    if (!utc) {
      ret = new stdDate(
        this.year,
        this.month - 1,
        this.day,
        this.hour,
        this.minute,
        this.second,
        this.microsecond / 1e3
      );
      ret.setFullYear(this.year);
    } else {
      ret = new stdDate(stdDate.UTC(
        this.year,
        this.month - 1,
        this.day,
        this.hour,
        this.minute,
        this.second,
        this.microsecond / 1e3
      ));
      ret.setUTCFullYear(this.year);
    }
    return ret;
  }
  /**
   * Return Date object with same year, month and day.
   * @returns {!Date}
   */
  date() {
    if (this["_date"] != null)
      return this["_date"];
    const ret = new Date(this.year, this.month, this.day);
    trySetProperty(this, "_date", ret);
    return ret;
  }
  /**
   * Return Time object with same hour, minute, second, microsecond and fold.
   * tzInfo is null.
   * @returns {!Time}
   */
  time() {
    if (this["_time"] != null)
      return this["_time"];
    const ret = new Time(
      this.hour,
      this.minute,
      this.second,
      this.microsecond,
      null,
      this.fold
    );
    trySetProperty(this, "_time", ret);
    return ret;
  }
  /**
   * Return Time object with same hour, minute, second, microsecond, fold, and
   * tzInfo attributes.
   * @returns {!Time}
   */
  timetz() {
    if (this["_timetz"] != null)
      return this["_timetz"];
    const ret = new Time(
      this.hour,
      this.minute,
      this.second,
      this.microsecond,
      this.tzInfo,
      this.fold
    );
    trySetProperty(this, "_timetz", ret);
    return ret;
  }
  /**
   * Return a DateTime with the same attributes, except for those attributes
   * given new values by whichever keyword arguments are specified. Note that
   * \{tzInfo: null\} can be specified to create a naive DateTime from an
   * aware DateTime with no conversion of date and time data.
   * @param {Object} newValues The object consisting of new values.
   * @param {number} [newValues.year]
   * @param {number} [newValues.month]
   * @param {number} [newValues.day]
   * @param {number} [newValues.hour]
   * @param {number} [newValues.minute]
   * @param {number} [newValues.second]
   * @param {number} [newValues.microsecond]
   * @param {?TZInfo} [newValues.tzInfo]
   * @param {number} [newValues.fold]
   * @returns {!DateTime}
   */
  replace({
    year = this.year,
    month = this.month,
    day = this.day,
    hour = this.hour,
    minute = this.minute,
    second = this.second,
    microsecond = this.microsecond,
    tzInfo = this.tzInfo,
    fold = this.fold
  }) {
    return new _DateTime(
      year,
      month,
      day,
      hour,
      minute,
      second,
      microsecond,
      tzInfo,
      fold
    );
  }
  /**
   * Return a DateTime object with new tzInfo attribute tz, adjusting the date
   * and time data so the result is the same UTC time as self, but in tz’s
   * local time.
   * If self is naive, it is presumed to represent time in the system timezone.
   * @param {?TZInfo} tz The target timezone. If null, the system local
   *                     timezone is assumed for the target timezone.
   * @returns {!DateTime}
   */
  asTimeZone(tz = null) {
    if (this.tzInfo === tz)
      return this;
    const offset = this.utcOffset();
    if (offset == null && tz == null)
      return this;
    let utc;
    if (offset == null) {
      const local = this.replace({ tzInfo: LOCALTZINFO });
      utc = sub(
        local,
        /** @type {TimeDelta} */
        local.utcOffset()
      );
    } else {
      utc = sub(this, offset);
    }
    const tmpTZ = tz != null ? tz : LOCALTZINFO;
    const ret = tmpTZ.fromUTC(utc.replace({ tzInfo: tmpTZ }));
    return ret.replace({ tzInfo: tz });
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.utcOffset(this).
   * @returns {?TimeDelta}
   */
  utcOffset() {
    return this.tzInfo == null ? null : this.tzInfo.utcOffset(this);
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.dst(this).
   * @returns {?TimeDelta}
   */
  dst() {
    return this.tzInfo == null ? null : this.tzInfo.dst(this);
  }
  /**
   * If tzInfo is null, returns null, else returns this.tzInfo.tzName(this).
   * @returns {?string}
   */
  tzName() {
    return this.tzInfo == null ? null : this.tzInfo.tzName(this);
  }
  /**
   * Return POSIX timestamp corresponding to the DateTime instance.
   * @returns {number}
   */
  timeStamp() {
    if (this["_timestamp"] != null)
      return this["_timestamp"];
    let dt = this;
    if (this.utcOffset() == null) {
      dt = this.replace({ tzInfo: LOCALTZINFO });
    }
    const ret = sub(dt, new _DateTime(
      1970,
      1,
      1,
      0,
      0,
      0,
      0,
      TimeZone.utc
    )).totalSeconds();
    trySetProperty(this, "_timestamp", ret);
    return ret;
  }
  /**
   * Return a string representing the date and time in ISO 8601 format.
   * @param {string} sep One-character separator placed between the date and
   *                     time portions of the result.
   * @param {"auto"|"microseconds"|"milliseconds"|"seconds"|"minutes"|"hours"
   * } timespec The number of additional components of the time to
   *            include.
   * @returns {string}
   */
  isoFormat(sep = "T", timespec = "auto") {
    return this.date().isoFormat() + sep + this.timetz().isoFormat(timespec);
  }
  /**
   * Return a string representing the date and time, controlled by an explicit
   * format string
   * @param {string} format The format string.
   * @returns {string}
   */
  strftime(format) {
    return strftime(this, format);
  }
  /**
   * Same as isoFormat(' ').
   * @returns {string}
   */
  toString() {
    return this.isoFormat(" ");
  }
  /**
   * Same as timeStamp().
   * @returns {number}
   */
  valueOf() {
    return this.timeStamp();
  }
  /**
   * The earliest representable DateTime, new DateTime(MINYEAR, 1, 1).
   * @type {!DateTime}
   */
  static get min() {
    if (!dateTimeMin) {
      dateTimeMin = new _DateTime(MINYEAR, 1, 1);
    }
    return dateTimeMin;
  }
  /**
   * The latest representable DateTime, new DateTime(MAXYEAR, 12, 31, 23, 59,
   * 59, 999999).
   * @type {!DateTime}
   */
  static get max() {
    if (!dateTimeMax) {
      dateTimeMax = new _DateTime(MAXYEAR, 12, 31, 23, 59, 59, 999999);
    }
    return dateTimeMax;
  }
  /**
   * The smallest possible difference between non-equal DateTime objects,
   * new TimeDelta(\{microseconds: 1\}).
   * @type {!TimeDelta}
   */
  static get resolution() {
    if (!dateTimeResolution) {
      dateTimeResolution = new TimeDelta({ microseconds: 1 });
    }
    return dateTimeResolution;
  }
};
function typeName(obj) {
  if (obj === null) {
    return "null";
  } else if (obj === void 0) {
    return "undefined";
  } else {
    return obj.constructor.name;
  }
}
function add(a, b) {
  function date_plus_timedelta(d, td) {
    d = d.toStdDate();
    d.setDate(d.getDate() + td.days);
    return Date.fromStdDate(d);
  }
  function datetime_plus_timedelta(dt, td) {
    const microseconds = divmod(dt.microsecond + td.microseconds, 1e3);
    const d = safeStdDateUTC(
      dt.year,
      dt.month,
      dt.day + td.days,
      dt.hour,
      dt.minute,
      dt.second + td.seconds,
      microseconds[0]
    );
    const ret = DateTime.fromStdDate(d, true).replace({
      tzInfo: dt.tzInfo
    });
    return ret.replace({ microsecond: ret.microsecond + microseconds[1] });
  }
  function time_plus_timedelta(t, td) {
    const total_microseconds = t.microsecond + td.microseconds;
    const total_seconds = t.second + t.minute * 60 + t.hour * 3600 + td.seconds;
    let [q, r] = divmod(total_microseconds, 1e6);
    const microsecond = r;
    [q, r] = divmod(total_seconds + q, 60);
    const second = r;
    [q, r] = divmod(q, 60);
    const minute = r;
    [q, r] = divmod(q, 24);
    const hour = r;
    return new Time(hour, minute, second, microsecond, t.tzInfo, t.fold);
  }
  if (a instanceof TimeDelta && b instanceof TimeDelta) {
    return new TimeDelta({
      days: a.days + b.days,
      seconds: a.seconds + b.seconds,
      microseconds: a.microseconds + b.microseconds
    });
  }
  if (a instanceof DateTime && b instanceof TimeDelta) {
    return datetime_plus_timedelta(a, b);
  }
  if (a instanceof TimeDelta && b instanceof DateTime) {
    return datetime_plus_timedelta(b, a);
  }
  if (a instanceof Date && b instanceof TimeDelta) {
    return date_plus_timedelta(a, b);
  }
  if (a instanceof TimeDelta && b instanceof Date) {
    return date_plus_timedelta(b, a);
  }
  if (a instanceof Time && b instanceof TimeDelta) {
    return time_plus_timedelta(a, b);
  }
  if (a instanceof TimeDelta && b instanceof Time) {
    return time_plus_timedelta(b, a);
  }
  throw new TypeDateTimeError(
    `Cannot add type "${typeName(a)}" and type "${typeName(b)}".`
  );
}
function sub(a, b) {
  if (a instanceof TimeDelta && b instanceof TimeDelta) {
    return new TimeDelta({
      days: a.days - b.days,
      seconds: a.seconds - b.seconds,
      microseconds: a.microseconds - b.microseconds
    });
  }
  if (a instanceof DateTime && b instanceof TimeDelta) {
    const microseconds = divmod(a.microsecond - b.microseconds, 1e3);
    const d = safeStdDateUTC(
      a.year,
      a.month,
      a.day - b.days,
      a.hour,
      a.minute,
      a.second - b.seconds,
      microseconds[0]
    );
    const ret = DateTime.fromStdDate(d, true).replace({
      tzInfo: a.tzInfo
    });
    return ret.replace({ microsecond: ret.microsecond + microseconds[1] });
  }
  if (a instanceof DateTime && b instanceof DateTime) {
    const aOffset = a.utcOffset();
    const bOffset = b.utcOffset();
    if (!(aOffset == null && bOffset == null) && a.tzInfo !== b.tzInfo) {
      if (aOffset == null || bOffset == null)
        throw new TypeDateTimeError(
          'Cannot subtract between naive "DateTime" and aware "DateTime"'
        );
      a = sub(a, aOffset);
      b = sub(b, bOffset);
    }
    const days = a.toOrdinal() - b.toOrdinal();
    return new TimeDelta({
      days,
      hours: a.hour - b.hour,
      minutes: a.minute - b.minute,
      seconds: a.second - b.second,
      microseconds: a.microsecond - b.microsecond
    });
  }
  if (a instanceof Date && b instanceof TimeDelta) {
    const d = a.toStdDate(true);
    d.setDate(d.getDate() - b.days);
    return Date.fromStdDate(d, true);
  }
  if (a instanceof Date && b instanceof Date) {
    return new TimeDelta({ days: a.toOrdinal() - b.toOrdinal() });
  }
  if (a instanceof Time && b instanceof TimeDelta) {
    return add(a, neg(b));
  }
  if (a instanceof Time && b instanceof Time) {
    const aOffset = a.utcOffset();
    const bOffset = b.utcOffset();
    if (!(aOffset == null && bOffset == null) && a.tzInfo !== b.tzInfo) {
      if (aOffset == null || bOffset == null)
        throw new TypeDateTimeError(
          'Cannot subtract between naive "Time" and aware "Time"'
        );
      a = sub(a, aOffset);
      b = sub(b, bOffset);
    }
    const aTimeDelta = new TimeDelta({
      hours: a.hour,
      minutes: a.minute,
      seconds: a.second,
      microseconds: a.microsecond
    });
    const bTimeDelta = new TimeDelta({
      hours: b.hour,
      minutes: b.minute,
      seconds: b.second,
      microseconds: b.microsecond
    });
    const ret = sub(aTimeDelta, bTimeDelta);
    return new TimeDelta({
      seconds: ret.seconds,
      microseconds: ret.microseconds
    });
  }
  throw new TypeDateTimeError(
    `Cannnot subtract type "${typeName(b)}" from type "${typeName(a)}".`
  );
}
function neg(a) {
  if (a instanceof TimeDelta) {
    return new TimeDelta({
      days: -a.days,
      seconds: -a.seconds,
      microseconds: -a.microseconds
    });
  }
  throw new TypeDateTimeError(`Cannot negate type "${typeName(a)}".`);
}
function cmp(a, b) {
  function _comp(a2, b2) {
    if (a2 === b2)
      return 0;
    if (a2 > b2)
      return 1;
    if (a2 < b2)
      return -1;
    throw new TypeError();
  }
  if (a instanceof TimeDelta && b instanceof TimeDelta) {
    let c = _comp(a.days, b.days);
    if (c)
      return c;
    c = _comp(a.seconds, b.seconds);
    if (c)
      return c;
    c = _comp(a.microseconds, b.microseconds);
    return c;
  }
  if (a instanceof DateTime && b instanceof DateTime) {
    let ta = a;
    let tb = b;
    const aOffset = ta.utcOffset(), bOffset = tb.utcOffset();
    if (!(aOffset == null && bOffset == null) && ta.tzInfo !== tb.tzInfo) {
      if (aOffset == null || bOffset == null)
        throw new TypeDateTimeError(
          'Cannot compare naive "DateTime" to aware "DateTime"'
        );
      ta = sub(ta, aOffset);
      tb = sub(tb, bOffset);
    }
    let c = _comp(ta.year, tb.year);
    if (c)
      return c;
    c = _comp(ta.month, tb.month);
    if (c)
      return c;
    c = _comp(ta.day, tb.day);
    if (c)
      return c;
    c = _comp(ta.hour, tb.hour);
    if (c)
      return c;
    c = _comp(ta.minute, tb.minute);
    if (c)
      return c;
    c = _comp(ta.second, tb.second);
    if (c)
      return c;
    c = _comp(ta.microsecond, tb.microsecond);
    if (c)
      return c;
    c = _comp(ta.fold, tb.fold);
    return c;
  }
  if (a instanceof Date && b instanceof Date) {
    return _comp(a.toOrdinal(), b.toOrdinal());
  }
  if (a instanceof Time && b instanceof Time) {
    let ta = a;
    let tb = b;
    const aOffset = ta.utcOffset(), bOffset = tb.utcOffset();
    if (!(aOffset == null && bOffset == null) && ta.tzInfo !== tb.tzInfo) {
      if (aOffset == null || bOffset == null)
        throw new TypeDateTimeError(
          'Cannot compare naive "Time" object to aware "Time" object'
        );
      ta = sub(ta, aOffset);
      tb = sub(tb, bOffset);
    }
    let c = _comp(ta.hour, tb.hour);
    if (c)
      return c;
    c = _comp(ta.minute, tb.minute);
    if (c)
      return c;
    c = _comp(ta.second, tb.second);
    if (c)
      return c;
    c = _comp(ta.microsecond, tb.microsecond);
    if (c)
      return c;
    c = _comp(ta.fold, tb.fold);
    return c;
  }
  throw new TypeDateTimeError(
    `Cannot compare type "${typeName(a)}" to type "${typeName(b)}".`
  );
}
function timedelta(...args) {
  return new TimeDelta(...args);
}
function date(...args) {
  return new Date(...args);
}
function timezone(...args) {
  return new TimeZone(...args);
}
function time(...args) {
  return new Time(...args);
}
function datetime(...args) {
  return new DateTime(...args);
}

// node_modules/@silane/datetime/src/expression.js
var DtexprDateTimeError = class extends DateTimeError {
  /**
   * @param {*[]} expression The expression caused this error.
   * @param {[number, number]} pos Position of the error in the expression.
   * @param {string} message Error message.
   */
  constructor(expression, pos, message) {
    super(message);
    this.expression = expression;
    this.pos = pos;
  }
  toString() {
    const [pos1, pos2] = this.pos;
    const expressionStr = this.expression.map(
      (x) => typeof x === "string" ? x : "${...}"
    ).join("");
    const pos = this.expression.slice(0, pos1).map(
      (x) => typeof x === "string" ? x.length : 6
    ).reduce((acc, cur) => acc + cur, 0) + pos2;
    return `[${this.constructor.name}] ${this.message}
${expressionStr}
${" ".repeat(pos)}^`;
  }
};
var SyntaxDtexprDateTimeError = class extends DtexprDateTimeError {
};
var ExecutionDtexprDateTimeError = class extends DtexprDateTimeError {
  /**
   * @param {*[]} expression The expression caused this error.
   * @param {[number, number]} pos Position of the error in the expression.
   * @param {?Error} originalError The original error object.
   * @param {string} message Error message.
   */
  constructor(expression, pos, originalError, message) {
    super(expression, pos, message);
    this.originalError = originalError;
  }
  toString() {
    if (this.originalError) {
      return `${super.toString()}
Original Error: ${this.originalError}`;
    } else {
      return super.toString();
    }
  }
};
var Variable = class {
  constructor(name) {
    this.name = name;
  }
};
var Node = class {
  constructor(pos) {
    this.pos = pos;
  }
  execute(context) {
    throw new NotImplementedDateTimeError();
  }
};
var VariableNode = class extends Node {
  constructor(variableName, pos) {
    super(pos);
    this.variableName = variableName;
  }
  execute(context) {
    if (this.variableName in context.variables) {
      return context.variables[this.variableName];
    } else {
      throw new ExecutionDtexprDateTimeError(
        [],
        this.pos,
        null,
        `Varibale "${this.variableName}" is not defined in the execution context`
      );
    }
  }
};
var NegNode = class extends Node {
  constructor(node, pos) {
    super(pos);
    this.node = node;
  }
  execute(context) {
    const value = this.node.execute(context);
    try {
      return neg(value);
    } catch (e) {
      if (e instanceof DateTimeError) {
        throw new ExecutionDtexprDateTimeError(
          [],
          this.pos,
          e,
          "Execution error in negation operator."
        );
      }
      throw e;
    }
  }
};
var BinaryNode = class extends Node {
  constructor(lhs, rhs, pos) {
    super(pos);
    this.lhs = lhs;
    this.rhs = rhs;
  }
};
var CommonBinaryNode = class extends BinaryNode {
  /** @type {string} */
  get operatorName() {
    throw new NotImplementedDateTimeError();
  }
  operate(leftValue, rightValue) {
    throw new NotImplementedDateTimeError();
  }
  execute(context) {
    const lVal = this.lhs.execute(context);
    const rVal = this.rhs.execute(context);
    try {
      return this.operate(lVal, rVal);
    } catch (e) {
      if (e instanceof DateTimeError) {
        throw new ExecutionDtexprDateTimeError(
          [],
          this.pos,
          e,
          `Execution error in ${this.operatorName} operator.`
        );
      }
      throw e;
    }
  }
};
var LesserNode = class extends CommonBinaryNode {
  get operatorName() {
    return "lesser-than";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) < 0;
  }
};
var LesserEqualNode = class extends CommonBinaryNode {
  get operatorName() {
    return "lesser-or-equal";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) <= 0;
  }
};
var EqualNode = class extends CommonBinaryNode {
  get operatorName() {
    return "equal";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) == 0;
  }
};
var NotEqualNode = class extends CommonBinaryNode {
  get operatorName() {
    return "not-equal";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) != 0;
  }
};
var GreaterNode = class extends CommonBinaryNode {
  get operatorName() {
    return "greater-than";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) > 0;
  }
};
var GreaterEqualNode = class extends CommonBinaryNode {
  get operatorName() {
    return "greater-or-equal";
  }
  operate(leftValue, rightValue) {
    return cmp(leftValue, rightValue) >= 0;
  }
};
var AddNode = class extends CommonBinaryNode {
  get operatorName() {
    return "addition";
  }
  operate(leftValue, rightValue) {
    return add(leftValue, rightValue);
  }
};
var SubNode = class extends CommonBinaryNode {
  get operatorName() {
    return "subtraction";
  }
  operate(leftValue, rightValue) {
    return sub(leftValue, rightValue);
  }
};
var ParsingStr = class {
  /**
   * @param {(string|number|!TimeDelta|!Date|!Time|!DateTime)[]} s
   */
  constructor(s) {
    this.s = s;
    this.pos1 = 0;
    this.pos2 = 0;
  }
  consumeIf(startstr) {
    const item = this.s[this.pos1];
    if (typeof item === "string") {
      if (item.slice(this.pos2).startsWith(startstr)) {
        this.pos2 += startstr.length;
        if (this.pos2 >= item.length) {
          ++this.pos1;
          this.pos2 = 0;
        }
        return startstr;
      } else {
        return "";
      }
    } else {
      return "";
    }
  }
  consumeWhile(chars) {
    const item = this.s[this.pos1];
    if (typeof item === "string") {
      let ret = "";
      for (; this.pos2 < item.length; ++this.pos2) {
        if (chars.includes(this.s[this.pos1][this.pos2]))
          ret += this.s[this.pos1][this.pos2];
        else
          break;
      }
      if (this.pos2 >= item.length) {
        ++this.pos1;
        this.pos2 = 0;
      }
      return ret;
    } else {
      return "";
    }
  }
  consumeIfVariable() {
    if (this.s[this.pos1] instanceof Variable) {
      return this.s[this.pos1++];
    } else {
      return null;
    }
  }
};
function whitespace(s) {
  s.consumeWhile(" \n");
}
function expr(s) {
  whitespace(s);
  const lhs = poly(s);
  const pos = [s.pos1, s.pos2];
  if (s.consumeIf("<=")) {
    whitespace(s);
    const rhs = expr(s);
    return new LesserEqualNode(lhs, rhs, pos);
  } else if (s.consumeIf("<")) {
    whitespace(s);
    const rhs = expr(s);
    return new LesserNode(lhs, rhs, pos);
  } else if (s.consumeIf("==")) {
    whitespace(s);
    const rhs = expr(s);
    return new EqualNode(lhs, rhs, pos);
  } else if (s.consumeIf("!=")) {
    whitespace(s);
    const rhs = expr(s);
    return new NotEqualNode(lhs, rhs, pos);
  } else if (s.consumeIf(">=")) {
    whitespace(s);
    const rhs = expr(s);
    return new GreaterEqualNode(lhs, rhs, pos);
  } else if (s.consumeIf(">")) {
    whitespace(s);
    const rhs = expr(s);
    return new GreaterNode(lhs, rhs, pos);
  } else {
    return lhs;
  }
}
function poly(s) {
  whitespace(s);
  const lhs = term(s);
  whitespace(s);
  const pos = [s.pos1, s.pos2];
  if (s.consumeIf("+")) {
    whitespace(s);
    const rhs = poly(s);
    return new AddNode(lhs, rhs, pos);
  } else if (s.consumeIf("-")) {
    whitespace(s);
    const rhs = poly(s);
    return new SubNode(lhs, rhs, pos);
  } else {
    return lhs;
  }
}
function term(s) {
  return negterm(s);
}
function negterm(s) {
  whitespace(s);
  const pos = [s.pos1, s.pos2];
  if (s.consumeIf("-")) {
    whitespace(s);
    const node = realexpr(s);
    return new NegNode(node, pos);
  } else {
    return realexpr(s);
  }
}
function realexpr(s) {
  whitespace(s);
  const pos = [s.pos1, s.pos2];
  if (s.consumeIf("(")) {
    whitespace(s);
    const lhs = poly(s);
    if (!s.consumeIf(")")) {
      throw new SyntaxDtexprDateTimeError(
        s.s,
        pos,
        'Expected ")".'
      );
    }
    return lhs;
  }
  const variable = s.consumeIfVariable();
  if (variable != null) {
    return new VariableNode(variable.name, pos);
  }
  throw new SyntaxDtexprDateTimeError(s.s, pos, "Unexpected token.");
}
function isExpressionTokenListSame(a, b) {
  if (a.length !== b.length)
    return false;
  return a.map((x, i) => [x, b[i]]).every(([aToken, bToken]) => {
    if (aToken === bToken)
      return true;
    return aToken instanceof Variable && bToken instanceof Variable && aToken.name === bToken.name;
  });
}
var expressionCache = [];
function dtexpr(strings, ...values) {
  var _a;
  const variables = values.map((x, i) => [new Variable(`var_${i}`), x]);
  let list = [strings[0]];
  variables.forEach((variable, i) => {
    list.push(variable[0]);
    list.push(strings[i + 1]);
  });
  list = list.filter((x) => !(typeof x === "string" && !x));
  let expression = ((_a = expressionCache.find(
    (x) => isExpressionTokenListSame(x[0], list)
  )) == null ? void 0 : _a[1]) ?? null;
  if (!expression) {
    expression = expr(new ParsingStr(list));
    expressionCache.push([list, expression]);
  }
  try {
    return expression.execute({ variables: Object.fromEntries(
      variables.map((x) => [x[0].name, x[1]])
    ) });
  } catch (e) {
    if (e instanceof ExecutionDtexprDateTimeError) {
      e.expression = list;
    }
    throw e;
  }
}
export {
  Date,
  DateTime,
  DateTimeError,
  DtexprDateTimeError,
  ExecutionDtexprDateTimeError,
  LOCALTZINFO,
  MAXYEAR,
  MINYEAR,
  NotImplementedDateTimeError,
  SyntaxDtexprDateTimeError,
  TZInfo,
  Time,
  TimeDelta,
  TimeZone,
  TypeDateTimeError,
  ValueDateTimeError,
  add,
  cmp,
  date,
  datetime,
  dtexpr,
  neg,
  sub,
  time,
  timedelta,
  timezone
};
//# sourceMappingURL=@silane_datetime.js.map
