import asyncio
import shlex
from importlib import import_module
from importlib_resources import files, as_file

from rooman.core import RoomanBase
from rooman.core import errors

from .jobs import new_job


async def run_external_program(cmd):
    splited = shlex.split(cmd)
    try:
        process = await asyncio.create_subprocess_exec(*splited)
    except FileNotFoundError as e:
        raise errors.RuntimeRoomanError() from e
    try:
        await asyncio.wait_for(process.wait(), 1)
    except asyncio.TimeoutError as e:
        process.kill()
        await process.wait()
        raise errors.RuntimeRoomanError() from e
    return process.returncode


class MyRooman(RoomanBase):
    def __init__(self, *, irsend_command='ir-send -c 1 -f "{SIGNAL_FILE}" 1'):
        super().__init__()
        self._irsend_command = irsend_command

    async def do_action(self, action_id, action_free_parameter):
        signal_file = {
            'light-full': 'light/full.txt',
            'light-on': 'light/on.txt',
            'light-mini': 'light/mini.txt',
            'light-off': 'light/off.txt',
            'light-brighter': 'light/brighter.txt',
            'light-darker': 'light/darker.txt',
        }.get(action_id)
        if signal_file is None:
            raise errors.ActionIDNotFoundError(action_id)

        with as_file(
            files(import_module('..data', __name__)) / 'irsignals' / signal_file
        ) as filepath:
            response = await run_external_program(
                self._irsend_command.format(SIGNAL_FILE=filepath)
            )
        return response

    async def do_create_job(self, job_type_id, new_job_free_parameter):
        return await new_job(self, job_type_id, new_job_free_parameter)
