from datetime import datetime, time, timezone
import numbers
from collections.abc import Mapping
import asyncio

from rooman.core import Job
from rooman.core import errors


class AlarmJob(Job):
    def __init__(self, rooman):
        self.rooman = rooman
        self.time = None
        self.alarm_task = None

    @staticmethod
    def get_time_parameter(free_parameter):
        if not isinstance(free_parameter, Mapping):
            raise errors.free_parameter_type_error([], free_parameter, Mapping)
        try:
            t = free_parameter['time']
        except KeyError:
            raise errors.free_parameter_key_error([], free_parameter, 'time')
        if not isinstance(t, numbers.Real):
            raise errors.free_parameter_type_error(['time'], t, numbers.Real)
        ti = int(t)
        if not (0 <= ti < 60 * 60 * 24):
            raise errors.free_parameter_value_error(['time'], t)
        return time(ti // 3600, (ti // 60) % 60, ti % 60)

    @classmethod
    async def new(cls, rooman, new_job_free_parameter):
        try:
            t = cls.get_time_parameter(new_job_free_parameter)
        except errors.FreeParameterError as e:
            raise errors.NewJobFreeParameterError(e.errors) from e
        ret = cls(rooman)
        await ret.set_time(t)
        return ret

    async def alarm(self, time):
        while True:
            now = datetime.now(time.tzinfo)
            today = datetime.combine(now.date(), time)
            delay = (today - now).total_seconds()
            if delay <= 0:
                delay += 60 * 60 * 24
            print('----------{}------------'.format(delay))
            await asyncio.sleep(delay)
            print('----------alarm------------')
            await self.rooman.invoke_action('light-full', None)
        
    async def set_time(self, time):
        await self.cancel_alarm_task()
        self.alarm_task = asyncio.ensure_future(self.alarm(time))
        self.time = time

    async def on_action(self, job_action_free_parameter):
        try:
            t = self.get_time_parameter(job_action_free_parameter)
        except errors.FreeParameterError as e:
            if all(x.path == ['time'] for x in e.errors):
                raise errors.JobActionFreeParameterError(e.errors) from e
        else:
            await self.set_time(t)
        t = self.time
        t = t.hour * 3600 + t.minute * 60 + t.second
        return {'time': t}
    
    async def on_delete(self):
        await self.cancel_alarm_task()
    
    async def cancel_alarm_task(self):
        if self.alarm_task is None:
            return
        self.alarm_task.cancel()
        try:
            await self.alarm_task
        except asyncio.CancelledError:
            pass
        finally:
            self.alarm_task = None

async def new_job(rooman, job_type_id, new_job_free_parameter):
    job_type_dict = {
        'alarm': AlarmJob
    }

    job_type = job_type_dict.get(job_type_id)
    if job_type is None:
        raise errors.JobTypeIDNotFoundError(job_type_id)

    job = await job_type.new(rooman, new_job_free_parameter)
    return job
